//
//  IDIDevice.h
//  iDeviceInterface
//
//  Created by Chris Devor on 7/14/11.
//  Copyright 2011 Macroplant. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "MobileDevice.h"
#import "IDDeviceConnectionManager.h"
#import "IDPathInfo.h"

extern NSString* AFCADDRESS;
extern NSString* AFC2ADDRESS;
extern NSString *SPRINGBOARD_SERVICE;


typedef enum {
    DEVICECOPYVALUE_ActivationState,
    DEVICECOPYVALUE_ActivationStateAcknowledged,
    DEVICECOPYVALUE_BasebandBootloaderVersion,
    DEVICECOPYVALUE_BasebandVersion,
    DEVICECOPYVALUE_BluetoothAddress,
    DEVICECOPYVALUE_BuildVersion,
    DEVICECOPYVALUE_DeviceCertificate,
    DEVICECOPYVALUE_DeviceClass,
    DEVICECOPYVALUE_DeviceColor,
    DEVICECOPYVALUE_DeviceName,
    DEVICECOPYVALUE_DevicePublicKey,
    DEVICECOPYVALUE_FirmwareVersion,
    DEVICECOPYVALUE_HostAttached,
    DEVICECOPYVALUE_IntegratedCircuitCardIdentity,
    DEVICECOPYVALUE_InternationalMobileEquipmentIdentity,
    DEVICECOPYVALUE_InternationalMobileSubscriberIdentity,
    DEVICECOPYVALUE_ModelNumber,
    DEVICECOPYVALUE_PhoneNumber,
    DEVICECOPYVALUE_ProductType,
    DEVICECOPYVALUE_ProductVersion,
    DEVICECOPYVALUE_ProtocolVersion,
    DEVICECOPYVALUE_RegionInfo,
    DEVICECOPYVALUE_SBLockdownEverRegisteredKey,
    DEVICECOPYVALUE_SIMStatus,
    DEVICECOPYVALUE_SerialNumber,
    DEVICECOPYVALUE_SomebodySetTimeZone,
    DEVICECOPYVALUE_TimeIntervalSince1970,
    DEVICECOPYVALUE_TimeZone,
    DEVICECOPYVALUE_TimeZoneOffsetFromUTC,
    DEVICECOPYVALUE_TrustedHostAttached,
    DEVICECOPYVALUE_UniqueDeviceID,
    DEVICECOPYVALUE_Uses24HourClock,
    DEVICECOPYVALUE_WiFiAddress,
    DEVICECOPYVALUE_iTunesHasConnected,
} DEVICECOPYVALUE;

NSString * const DEVICECOPYVALUE_toString[50];

/**
 Data class for info on the device's app directories
 */
@interface IDDeviceAppDirectoryItem : NSObject {
    
    NSString *name;
    NSString* ID;
    NSString* iconPath;
    NSString* appFolder;
    NSArray* iCloudDocumentIDs;
    NSString* iCloudKVID;
    BOOL isFileSharingEnabled;
    
    NSDictionary* infoDictionary;
}
@property(readwrite, retain) NSString *name;
@property(readwrite, retain) NSString *ID;
@property(readwrite, retain) NSString *iconPath;
@property(readwrite, retain) NSString *appFolder;
@property(readwrite, retain) NSArray* iCloudDocumentIDs;
@property(readwrite, retain) NSString* iCloudKVID;
@property(readwrite)         BOOL      isFileSharingEnabled;
@property(readwrite, copy) NSDictionary* infoDictionary;
@property(readwrite)         BOOL       isFromAppStore;

@end

@class IDDeviceThread;

/**
 This represents our connection to an IOS device that is plugged in to the computer (and hopefully eventually, connected via WiFi).
 Use this to perform all file system tasks.
 */
@interface IDIDevice : NSObject {
    
    
    // Command to cancel current operations. Also set abortForDisconnect if the reason is that the device was disconnected
    BOOL abortNow;
    BOOL abortForDisconnect;
    BOOL operationActive;
    
    struct am_device *myDevice;
    
    
	struct afc_connection *myConnection;
    NSString* currentService;
    
    // We may choose to leave a file open; if we do, then this is left with that value (otherwise it's reset to nil)
    NSString* openPath;
    int openFileMode;
    afc_file_ref openFileRef;
    
    NSLock* cancelCopyLock;
    NSString* cancelCopyDevicePath;
    NSString* cancelCopyDeviceService;
    NSString* cancelCopyLocalPath;
	
	BOOL isJailbreaked;
    BOOL isConnected;
    
    IDDeviceThread* connectionThread;
    NSLock* threadCancelLock;
    
    NSString* deviceName;
    NSString* uniqueDeviceID;
    NSString *_productVersion;
    NSMutableArray* appBundles;
    
    NSDictionary* deviceInfoDict;
    
    NSImage* deviceImage;
    
    
    NSObject <IDDeviceConnectionManagerDelegate> *delegate;
    IDDeviceConnectionManager* connectionManager;
    
    NSDictionary *deviceCopyValues;
}

/** Reference to the MobileDevice object for the device connection */
@property(readonly) struct am_device *myDevice;
//@property(readonly) struct afc_connection *myConnection;

/** YES if it's a jailbroken device, NO if not */
@property(readonly) BOOL isJailbreaked;

/** YES if we are currently connected; although once disconnected, this object will be removed from the manager, so the
 state shouldn't be NO for very long. */
@property(readonly) BOOL isConnected;

/** The currently active service. You don't *have* to know this, but keep in mind that switch services takes a bit of time, so you don't want to be doing it a lot. */
@property (readonly) NSString* currentService;

// Do not call these directly, to cancel use the cancelOperation method

/** DO NOT USE. Used by the device manager */
@property BOOL abortNow;

/** DO NOT USE. Used by the device manager */
@property BOOL abortForDisconnect;

/** DO NOT USE. Used by the device manager */
@property BOOL operationActive;

/** DO NOT USE. Used by the device manager */
@property (assign) IDDeviceConnectionManager* connectionManager;


//@property (readonly) NSString* deviceName;
//@property (readonly) NSString* activationState;
//@property (readonly) NSString* activationStateAcknowledged;
//@property (readonly) NSString* basebandBootloaderVersion;
//@property (readonly) NSString* basebandVersion;
//@property (readonly) NSString* bluetoothAddress;
//@property (readonly) NSString* buildVersion;
//@property (readonly) NSString* deviceCertificate;
//@property (readonly) NSString* deviceClass;
//@property (readonly) NSString* devicePublicKey;
//@property (readonly) NSString* firmwareVersion;
//@property (readonly) NSString* fostAttached;
//@property (readonly) NSString* integratedCircuitCardIdentity;
//@property (readonly) NSString* internationalMobileEquipmentIdentity;
//@property (readonly) NSString* internationalMobileSubscriberIdentity;
//@property (readonly) NSString* modelNumber;

/** If it's an iPhone, its phone number */
@property (readonly) NSString* phoneNumber;
//@property (readonly) NSString* productType;
@property (readonly) NSString* productVersion;
//@property (readonly) NSString* protocolVersion;
//@property (readonly) NSString* regionInfo;
//@property (readonly) NSString* sBLockdownEverRegisteredKey;
//@property (readonly) NSString* sIMStatus;
//@property (readonly) NSString* serialNumber;
//@property (readonly) NSString* somebodySetTimeZone;
//@property (readonly) NSString* timeIntervalSince1970;
//@property (readonly) NSString* timeZone;
//@property (readonly) NSString* timeZoneOffsetFromUTC;
//@property (readonly) NSString* trustedHostAttached;

/** Each device has a factory-set unique ID */
@property (readonly) NSString* uniqueDeviceID;
//@property (readonly) NSString* uses24HourClock;
//@property (readonly) NSString* wiFiAddress;
//@property (readonly) NSString* iTunesHasConnected;

/** Only the device manager init's devices, so don't worry about this one */
- (id)initWithDevice:(struct am_device *)thisDevice allowAFC2:(BOOL) allowAFC2 delegate:(NSObject <IDDeviceConnectionManagerDelegate>*) del;

/**
 Copy a file or folder from the device to the local computer
 
 @param sourcePathOnDevice - path to file or folder on device
 @param destinationPathOnComputer - destination file/folder path and name on the local machine
 @param service - the service connection
 @return YES if succeeded, otherwise NO
 
 @discussion The work actually happens on a background thread, but this will wait until that finishes before returning.
             If the item is a folder, the destination folder will be created, and all it's contents will be recursively copied.
 */
- (BOOL) pathCopyToComputerFromDevice:(NSString *)sourcePathOnDevice destinationOnComputer:(NSString *)destinationPathOnComputer serviceName:(NSString*)service;

/**
 Copy a file or folder from the local computer to the device
 
 @param sourcePathOnComputer - path to file or folder on local computer
 @param targetPathOnDevice - destination file/folder path and name on the device
 @param service - the service connection
 @return YES if succeeded, otherwise NO
 
 @discussion The work actually happens on a background thread, but this will wait until that finishes before returning.
             If the item is a folder, the destination folder will be created, and all it's contents will be recursively copied.
 */
- (BOOL) pathCopyToDeviceFromComputer:(NSString *)sourcePathOnComputer targetPath:(NSString *) targetPathOnDevice serviceName:(NSString*)service;

/**
 Read all the data from a file, and return it in an NSData.
 @param sourcePathOnDevice - path to file on device
 @param service - name of service connection
 @return NSData with the file contents, or nil if failed or not a file
 */
- (NSData*) readDataFromFile:(NSString *)sourcePathOnDevice serviceName:(NSString*)service;

/**
 Read part of the data from a file, and return it in an NSData.
 @param sourcePathOnDevice - path to file on device
 @param offset - number of bytes into the file to start
 @param length - number of bytes to copy
 @param service - name of service connection
 @return NSData with the file contents, or nil if failed or not a file
 */
- (NSData*) readDataFromFile:(NSString *)sourcePathOnDevice offset:(unsigned long long)offset length:(unsigned long long)length serviceName:(NSString*)service;

/**
 Write some data to a file on the device.
 
 @param data - the data to write. The file will then have only this data, its original contents will be lost
 @param pathOnDevice - path to the file on the device. If it didn't already exist, it wil be created.
 @param service - name of the service connection
 @return YES if succeeded, NO if failed
 */
- (BOOL) writeData:(NSData*)data toFile:(NSString*)pathOnDevice serviceName:(NSString*)service;

/**
 Append data to an existing file.
 
 @param data - the data to write to the file
 @param range - the position within the file to write
 @param pathOnDevice - path to the file on the device. The file must already exist
 @param service - name of the service connection
 @return YES if succeeded, NO if failed for any reason
 
 @discussion You should only count on this appending bytes at the given position. The behavior for writing bytes
             in the middle of the file in undifined, but most likely any bytes from that to the original end will
             be lost.
 */
- (BOOL) writeData:(NSData*)data range:(NSRange)range toFile:(NSString*)pathOnDevice serviceName:(NSString*)service;

/**
 Cancel whatever operation is currently happening.
 */
- (void) cancelOperation;

/**
 Get an array of IDDeviceAppDirectoryItem's for the various apps we can connect to.
 
 @return list of app connection info's
 
 @discussion This make take a few seconds the first time it is called. After that, the info is cached and never re-polled.
 */
- (NSArray*) appBundles;

/**
 Get the raw file/folder info for the given item
 
 @param path - path to the file, folder, or link
 @param service - name of the service connection
 @return Dictionary of raw info, as it comes to us from MobileDevice
 */
- (NSDictionary *) GetPathInfo:(NSString *)path serviceName:(NSString*)service;

/**
 Return a new IDPathInfo object for the given file/folder
 @param path - path to the file, folder, or link
 @param service - name of the service connection
 @return The new IDPathInfo for the item
 */
- (IDPathInfo *) GetIDPathInfo:(NSString *)path serviceName:(NSString*)service;

/**
 Try to get the image for the given app. Will return nil on failure.
 
 @param service - service connection for the app
 @return The image if it could be obtained, otherwise nil
 */
- (NSImage*) loadAppImageForApp:(NSString*)service;

/**
 Get an image for the device
 @return the image for the device; will always return something, defaults to a basic device if the correct one is unknown
 */
- (NSImage*) deviceImage;

/**
 Return a path to the PNG for the device image
 @return Path to the PNG used in deviceImage
 */
- (NSString*) deviceImagePath;

/**
 Is there a camera on this iOS device
 @return YES if there's at least one camera
 */
- (BOOL) deviceHasCamera;

/**
 Is the device an iPhone
 @return YES if an iPhone, NO if any other iOS device
 */
- (BOOL) deviceHasPhone;

/**
 Create a new folder at the given path.
 @param folderPath - path and folder name to create. Will create any parent folders needed as well
 @param service - name of the service connection
 @return YES on success
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) createNewFolder:(NSString *)folderPath serviceName:(NSString*)service;

/**
 Rename a given folder or file
 
 @param folderPath - full path for the item, including its name
 @param newFolderName - new name of just the item itself (not the full path)
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) renameFolder:(NSString *)folderPath newFolderName:(NSString *)newFolderName serviceName:(NSString*)service;

/**
 Move the given item to a new place, possibly renaming as well (this can be used to rename)
 
 @param sourcePath - full path and name for the item to move
 @param destPath - new full path and name for the item, on the same service connection
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
             This can only move an item on the same service connection.
 */
- (BOOL) movePath:(NSString*)sourcePath toPath:(NSString*)destPath serviceName:(NSString*)service;

/**
 Delete a folder or file.
 
 @param folderPath - path to th folder or file
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
             If the item is a folder, this will recursively delete all contents before deleting the folder itself.
 */
- (BOOL) removePath:(NSString *) folderPath serviceName:(NSString*)service;

/**
 Does the path exist on the device and service connection
 
 @param thisPath - path to a file or folder to check
 @param service - name of the service connection
 @param isDir - set to YES if the item is a directory, NO if it's a file, but left alone if it didn't exist in the first place
 @return YES if the path exists
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) pathExistsOnDevice:(NSString *) thisPath serviceName:(NSString *)service isDirectory:(BOOL *) isDir;

/**
 Does the path exist on the device and service connection
 
 @param thisPath - path to a file or folder to check
 @param service - name of the service connection
 @return YES if the path exists
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) pathExistsOnDevice:(NSString *) thisPath serviceName:(NSString *)service;
//- (BOOL) fileExistsOnDevice:(NSString *) thisPath serviceName:(NSString*)service;
//- (BOOL) directoryExistsOnDevice:(NSString *)thisPath serviceName:(NSString*)service;

/**
 Create a symbolic or hard link.
 
 @param path - full path and name of the link item
 @param target - full path (on the same service connection) of the target file/folder for the link
 @param service - name of the service connection
 @param isSymbolic - YES to make a symbolic link, NO to make it a hard link
 @return YES on success, NO if failed for any reason
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) createLinkAtPath:(NSString*)path targetPath:(NSString*)target serviceName:(NSString*)service isSymbolic:(BOOL)isSymbolic;
//- (BOOL) createSymbolicLinkAtPath:(NSString*)path targetPath:(NSString*)target serviceName:(NSString*)service;

/**
 Is the item a directory
 
 @param path - path to the item
 @param service - name of the service connection
 @return YES if it's a directory, NO if it's not or if it doesn't exist
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) isDirectory:(NSString *)path serviceName:(NSString*)service;

/**
 Get the file size in bytes of the file
 
 @param path - full path to the file
 @param service - name of the service connection
 @return The file size in bytes (use unsignedLongLongValue)
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (NSNumber *) getFileSize:(NSString *)path serviceName:(NSString*)service;

/**
 Set the file size of a file
 
 @param size - the new file size
 @param path - the path for the file
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (BOOL) setFileSize:(unsigned long long)size forPath:(NSString*)path serviceName:(NSString*)service;

/**
 Get the names of the items in a directory.
 
 @param directoryPath - path to the directory
 @param includeHidden - YES to include hidden items in the list, NO to leave them out
 @param service - name of the service connection
 @return array of NSString's of the file names, or nil on failure
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (NSArray *) contentsInDirectory:(NSString *)directoryPath includeHiddenFiles:(BOOL)includeHidden serviceName:(NSString*)service;

/**
 Get the names of the items in a directory, sorted in alphabetical order.
 
 @param directoryPath - path to the directory
 @param includeHidden - YES to include hidden items in the list, NO to leave them out
 @param service - name of the service connection
 @return array of NSString's of the file names, or nil on failure
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (NSArray *) sortedContentsInDirectory:(NSString *)directoryPath includeHiddenFiles:(BOOL)includeHidden serviceName:(NSString*)service;

/**
 Get the names of just the folders in a directory.
 
 @param directoryPath - path to the directory
 @param includeHidden - YES to include hidden items in the list, NO to leave them out
 @param service - name of the service connection
 @return array of NSString's of the file names, or nil on failure
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (NSArray *) getDirectories:(NSString *)directoryPath includeHidden:(BOOL)includeHidden serviceName:(NSString*)service;

/**
 Get the names of jsut the files in a directory.
 
 @param directoryPath - path to the directory
 @param includeHidden - YES to include hidden items in the list, NO to leave them out
 @param service - name of the service connection
 @return array of NSString's of the file names, or nil on failure
 
 @discussion The work is performed on a background thread, but this will wait until it finishes before returning.
 */
- (NSArray *) getFiles:(NSString *)directoryPath includeHidden:(BOOL)includeHidden serviceName:(NSString*)service;

/**
 Get the whole dictionary of device info, which includes things like firmware version, device type, various IDs...
 @return dictionary of device info.
 @discussion This is only polled once, the first time this is called. After that, we just return the same saved dictionary
 */
- (NSDictionary *) getDeviceInfo;

/**
 Get the total file system capacity
 @return device's file system size, in bytes, or nil if it could not be found
 */
- (NSNumber *) getDeviceCapacity;

/**
 Get the free space in the file system
 @return device's file system's free space, in bytes, or nil if it could not be found
 */
- (NSNumber *) getDeviceFreeSpace;

/**
 Get the size of a file on the device
 @param path - path to the file
 @param service - name of the service connection
 @return file size in bytes (use unsignedLongLongValue)
 */
- (NSNumber *) getFileSize:(NSString *)path serviceName:(NSString*)service;

/**
 Get the modification date for a file or folder on the device
 @param path - path to the file or folder
 @param service - name of the service connection
 @return The modification date, or nil if it could not be found or the file does not exist
 */
- (NSDate *) getDateModified:(NSString *)path serviceName:(NSString*)service;


/**
 Open a file for reading, and leave it open.
 @param path - path to the file
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
             close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
             interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (BOOL) openHangingFileForReading:(NSString*)path serviceName:(NSString*)service;

/**
 Open a file for writing, and leave it open.
 @param path - path to the file
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
 close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
 interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (BOOL) openHangingFileForWriting:(NSString*)path serviceName:(NSString*)service;

/**
 Open a file for reading and writing, and leave it open.
 @param path - path to the file
 @param service - name of the service connection
 @return YES on success, NO if failed for any reason
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
 close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
 interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (BOOL) openHangingFileForReadingAndWriting:(NSString*)path serviceName:(NSString*)service;

/**
 Close a hanging file
 @param path - path to the file
 @param service - name of the service connection
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
 close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
 interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (void) closeHangingFileRef:(NSString*)path serviceName:(NSString*)service;

/**
 Read from a hanging file.
 @param path - path to the file
 @param service - name of the service connection
 @param offset - byte offset to start reading in the file
 @param size - number of bytes to read
 @return On success, the data is returned. On end-of-file, an empty NSData is returned. Nil is returned on error.
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
 close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
 interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (NSData*) readHangingFile:(NSString*)path serviceName:(NSString*)service offset:(unsigned long long)offset size:(unsigned long long)size;

/**
 Write to a hanging file
 @param path - path to the file
 @param service - name of the service connection
 @param offset - number of bytes into the file to start writing
 @param size - number of bytes to write
 @param pData - pointer to the byte array to copy from
 @return The number of bytes actually written. Returns -1 on error
 
 @discussion Opening and reading/writing hanging files is really just a hint for optimization. The system will automatically
 close and re-open file as needed, if such operation interrupt the hanging file operations. However, if not
 interrupted, using hanging references is much faster than trying to accomplish the same thing with other operations.
 */
- (signed long long) writeHangingFile:(NSString*)path serviceName:(NSString*)service offset:(unsigned long long)offset size:(unsigned long long)size data:(const void*)pData;

/**
 Get a list of IDPathInfo's for the given folder
 
 @param path - path to the folder
 @param service - name of the service connection
 @return an array of IDPathInfo's for the contents of the folder, or nil on error
 */
- (NSArray *) getChildPathInfosWithPath:(NSString *)path serviceName:(NSString*)service;

/**
 Delegate for important events for the device
 @return the delegate
 */
- (NSObject <IDDeviceConnectionManagerDelegate> *) delegate;

/**
 Make sure the folder tree to the given path exists on the local machine
 @param destinationPath - the full path to prepare
 @return YES on success, NO if failed for any reason
 */
- (BOOL) prepareTargetPathOnLocalMachine:(NSString *) destinationPath;

/**
 Get a device value
 @param dcv - the device value to get
 @return the value of the key; returns nil if not found
 */
- (NSString *) deviceCopyValue:(DEVICECOPYVALUE) dcv;

/**
 Get a device value by its key.
 @param valKey  The key for the device value
 @return the value, or nil if it wasn't found
 */
- (NSString*) deviceCopyValueKey:(NSString*)valKey;

/**
 The user-set device name (like Jim Bob's iPhone)
 @return the device name
 */
- (NSString*) deviceName;

/**
 Cancel the given copy operation (either from device to local, or from local to device), if it is active.
 @param path - the path on the device (from or to)
 @param service - the name of the service connection
 @param localPath - the path on the local machine (from or to)
 
 @discussion If the copy operation is active or pending, this *should* cancel it right away.
 */
- (void) cancelCopyDevicePath:(NSString*)path deviceService:(NSString*)service localPath:(NSString*)localPath;

/**
 Ask the the springboard service for the icon for the app specified by appID
 @param appID - identifier of the app for which the icon should be retreived e.g. com.apple.Pages
 @return - PNG data for the app icon
 */
- (NSData *) springboardIconPngDataForApp:(NSString *)appBundleId;

@end
