//
//  IDDeviceConnectionManager.h
//  iDeviceInterface
//
//  Created by Chris Devor on 7/14/11.
//  Copyright 2011 Macroplant. All rights reserved.
//

#import <Foundation/Foundation.h>

@class IDIDevice;

typedef enum _IDIDeviceErrorCode {
    IDIDeviceErrorCode_Unknown,
    
    IDIDeviceErrorCode_AFCRead,   // Error reading file from device
    IDIDeviceErrorCode_AFCWrite,  // Error writing to device file
    IDIDeviceErrorCode_FSRead,    // Error reading from local file system
    IDIDeviceErrorCode_FSWrite    // Error writing to local file system
} IDIDeviceErrorCode;

typedef struct _IDIDeviceCopyStatus {
    __unsafe_unretained IDIDevice* device;
    __unsafe_unretained NSString* devicePath;
    __unsafe_unretained NSString* deviceService;
    __unsafe_unretained NSString* localPath;
    float fractionComplete;
    BOOL fromDeviceToComputer; // Else, from computer to device
} IDIDeviceCopyStatus;



/**
 The protocol for recieving events about devices
 */
@protocol IDDeviceConnectionManagerDelegate

/**
 A device has just connected.
 @param iDevice - the newly connected device
 */
- (void) handleDeviceConnected:(IDIDevice *) iDevice;

/**
 A device has just disconnected
 @param iDevice - the device that just disconnected
 
 @discussion The manager's reference to this device will be removed, and the IDIDevice object will be informed that it
 is no longer connected. However, to avoid any potential problems you should remove all of your own references
 to it right away, and let it destruct
 */
- (void) handleDeviceDisconnected:(IDIDevice *) iDevice;

/**
 Called during time-consuming operations.
 Pass the message along to the app's busy status manager
 @param busyStatus - the busy status message
 */
- (void) addBusyStatus:(NSString*) busyStatus;

/**
 Called during time-consuming operations.
 Pass the message along to the app's busy status manager
 @param busyStatus - the busy status message
 */
- (void) removeBusyStatus:(NSString*) busyStatus;

/**
 Called during time-consuming operations.
 Pass the message along to the app's busy status manager
 @param precentComplete - fraction of operation completed, 0.0-1.0
 @param busyStatus - the busy status message
 */
- (void) updateProgressToValue:(float)precentComplete forStatus:(NSString*) busyStatus;


// This is to replace the above addBusyStatus:, removeBusyStatus:, and updateProgressToValue:forStatus: above.
// This simply status the copy operation taking place, and let's the linking app's device delegate decide
// what to do about it.
- (void) updateFileCopy:(const IDIDeviceCopyStatus*)pStatus;

@optional

/**
 domain will be "com.macroplant.iDevice"
 localizedDescription will be filled in, if you wish to display it.
 code values are listed in the enum above
 
 @param error - the error
 */
- (void) handleError:(NSError*)error;

@end

/**
 This singleton manager deals with the communication with MobileDevice, to detect and iOS device connection and
 disconnection.
 */
@interface IDDeviceConnectionManager : NSObject {
    
    bool allowAFC2Connections;
    NSMutableArray* ma_connectedDevices;
    NSObject <IDDeviceConnectionManagerDelegate> *delegate;
    
    // Whether or not to double-check that a transfer from the device actually transferred correctly.
    // Note that this value will be copied before an actual file transfer, so you can safely change it in the middle
    // of a transfer, and it simply will not take effect until the next tranfer.
    BOOL validateTransfersFromDevice;
    
    // For handling file copy update status. The status is not handled right away, but rather queued and handled on the main thread
    NSMutableArray* ma_copyStatusQueue;
    BOOL isHandlingStatusUpdate;
}

// NOTE: this is left in for now for compatability, however please change to use the setup* class method instead
- (id)initWithDelegate:(NSObject <IDDeviceConnectionManagerDelegate>*)del AllowAFC2Connections:(BOOL) allowAFC2;

/**
 This should be called once during your applicationDidFinishLaunching setup, to tell the manager who to talk to
 when things happen.
 
 @param del - the connection delegate, what gets informed of device connections and other important events
 @param allowAFC2 - pass YES to allow jailbroken connections, NO to only allow regular AFC
 @return the singleton manager
 */
+ (IDDeviceConnectionManager*) setupWithDelegate:(NSObject <IDDeviceConnectionManagerDelegate>*)del AllowAFC2Connections:(BOOL) allowAFC2;

/**
 The singleton manager
 @return the singleton manager
 */
+ (IDDeviceConnectionManager *) sharedInstance;


// Atomic and read-write
@property BOOL validateTransfersFromDevice;

@end
